package gov.va.genisis2.dao.impl;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.common.enums.UserStatusEnum;
import gov.va.genisis2.dao.IStudyApprovalDao;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.StudyApproval;

import java.util.List;

import javax.persistence.Query;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class StudyApprovalDao.
 *
 * The StudyApproval data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, StudyApproval DAO
 * provide some specific data operations without exposing details of the
 * database.
 */
@Repository
@Transactional(value = "transactionManager", rollbackFor = Exception.class)
public class StudyApprovalDao extends AbstactHibernateDao implements IStudyApprovalDao {
	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(StudyApprovalDao.class);
	private static final String STUDYAPPROVALSTATUS_DISABLE_UPDATE_SQL = "UPDATE StudyApproval SET Active = :disableStatus, ModifiedOn = GETDATE() WHERE title  IN (:usersList) AND Active = :activeStatus";
	@Autowired
	private SessionFactory sessionFactory;

	/**
	 * This method is used to createStudyApproval.
	 * 
	 * @param entity
	 *            the entity.
	 * @return int This returns entity id.
	 */
	@Override
	public int createStudyApproval(StudyApproval entity) throws GenisisDAOException {
		if (entity == null) {
			return 0;
		}
		Integer studyApproval = this.maxRowValue();
		if (studyApproval == 0) {
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("Exception occurred while getting Max value on Study Approval table.Unable to create Study Approval");
			}
			return 0;
		} else
			entity.setId(studyApproval.intValue());
		try {
			Session session = sessionFactory.getCurrentSession();
			studyApproval = (Integer) session.save(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on createStudyApproval.", ex);
			throw new GenisisDAOException("Exception occurred on createStudyApproval.", ex);
		}
		return studyApproval.intValue();
	}

	/**
	 * This method is used to updateStudyApproval.
	 * 
	 * @param entity
	 * 
	 *            the entity.
	 * 
	 * @return int This returns entity id.
	 */
	@Override
	public int updateStudyApproval(StudyApproval entity) throws GenisisDAOException {
		if ((entity == null) || (this.getStudyApprovalsByID(entity.getId()) == null)) {
			return 0;
		}
		try {
			Session session = sessionFactory.getCurrentSession();
			session.update(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on updateStudyApproval.", ex);
			throw new GenisisDAOException("Exception occurred on updateStudyApproval.", ex);
		}
		StudyApproval studyApproval = this.getStudyApprovalsByID(entity.getId());
		return (null != studyApproval) ? studyApproval.getId() : null;
	}

	/**
	 * This method is used to getStudyApprovalsByID.
	 * 
	 * @param id
	 * 
	 *            the id.
	 * @return StudyApproval This returns StudyApproval.
	 */
	@Override
	public StudyApproval getStudyApprovalsByID(int id) throws GenisisDAOException {
		StudyApproval studyApproval = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			id=Integer.hashCode(id);
			studyApproval = session.get(StudyApproval.class, id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getStudyApprovalsByID.", ex);
			throw new GenisisDAOException("Exception occurred while querying getStudyApprovalsByID.", ex);
		}
		return studyApproval;
	}

	/**
	 * This method is used to getStudyApprovalsByUID.
	 * 
	 * @param uid
	 * 
	 *            the uid.
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovalsByUID(String uid) throws GenisisDAOException {
		List<StudyApproval> listStudyApproval = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<StudyApproval> query = criteriaBuilder.createQuery(StudyApproval.class);
			Root<StudyApproval> root = query.from(StudyApproval.class);
			Predicate condition = criteriaBuilder.equal(root.get("createdBy"), uid);
			query.where(condition);
			listStudyApproval = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getStudyApprovalsByUID.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getStudyApprovalsByUID.", ex);
		}
		return listStudyApproval;
	}

	/**
	 * This method is used to getStudyApprovals.
	 * 
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovals() throws GenisisDAOException {
		List<StudyApproval> listStudyApproval = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<StudyApproval> query = criteriaBuilder.createQuery(StudyApproval.class);
			Root<StudyApproval> root = query.from(StudyApproval.class);
			Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.STUDY_APPROVAL_ACTIVE.getText()), UserStatusEnum.ACTIVE.getId());
			Predicate condition2 = criteriaBuilder.notEqual(root.get(CommonEnum.STUDY_APPROVAL_TITLE.getText()), CommonEnum.STUDY_APPROVAL_TITLE_OPEARTIONS.getText());
			Predicate conditions = criteriaBuilder.and(condition1, condition2);
			query.where(conditions);
			query.select(root);
			listStudyApproval = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getStudyApprovals.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getStudyApprovals.", ex);
		}
		return listStudyApproval;
	}

	@Override
	public StudyApproval saveOrUpdate(StudyApproval studyApproval) throws GenisisDAOException {
		if (studyApproval == null) {
			throw new GenisisDAOException("studyApproval entity is null");
		}
		StudyApproval persistentStudyApproval;
		try {
			Session session = sessionFactory.getCurrentSession();
			if (studyApproval.getId() <= 0) {
				//studyApproval.setId(getMaxRowValue(session, StudyApproval.class, CommonEnum.ID.getText()));
				studyApproval.setId(this.maxRowValue());
			}
			persistentStudyApproval = (StudyApproval) updateData(session, studyApproval);
		} catch (Exception ex) {
			LOGGER.error("Exception occured on createuser.", ex);
			throw new GenisisDAOException("Exception occured on createuser.", ex);
		}
		return persistentStudyApproval;
	}

	/**
	 * Max row value.
	 *
	 * @return int This returns StudyApproval id.
	 */
	private int maxRowValue() {
		// TODO: createCriteria Deprecated
		int maxRowVal = 1;
		Integer result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			@SuppressWarnings("deprecation")
			Criteria criteria = session.createCriteria(StudyApproval.class).setProjection(Projections.max("id"));
			result = (Integer) criteria.uniqueResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on maxRowValue().", ex);
		}
		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}

	@Override
	public List<StudyApproval> getStudyApprovalsByTitle(String title) throws GenisisDAOException {
		List<StudyApproval> listStudyApproval = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<StudyApproval> query = criteriaBuilder.createQuery(StudyApproval.class);
			Root<StudyApproval> root = query.from(StudyApproval.class);
			Predicate condition = criteriaBuilder.equal(root.get("title"), title);
			query.where(condition);
			listStudyApproval = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getStudyApprovalsByTitle.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getStudyApprovalsByTitle.", ex);
		}
		return listStudyApproval;
	}

	@Override
	public int updateStudyApprovalStatus(List<String> studyApprovalList) throws GenisisDAOException {
		int result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			Query query = session.createQuery(STUDYAPPROVALSTATUS_DISABLE_UPDATE_SQL);
			query.setParameter("disableStatus", UserStatusEnum.DISABLE.getId());
			query.setParameter("usersList", studyApprovalList);
			query.setParameter("activeStatus", UserStatusEnum.ACTIVE.getId());
			result = query.executeUpdate();
		} catch (Exception ex) {
			LOGGER.error("Exception occured on updateStudyApprovalStatus", ex);
			throw new GenisisDAOException("Exception occured on updateStudyApprovalStatus", ex);
		}
		return result;
	}
}
